#ifndef V3D_BLOBS_SCANLINES_H_INCLUDED
#define V3D_BLOBS_SCANLINES_H_INCLUDED


namespace V3D {


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


// Definit un intervalle a l'interieur d'une ligne (valeurs entieres)
class ScanLineRange
{
public:
	int32 nRangeBeg;
	int32 nRangeEnd;

	inline ScanLineRange() {}
	inline ScanLineRange( int32 beg, int32 end) : nRangeBeg(beg), nRangeEnd(end) {}	
	inline ScanLineRange( const ScanLineRange& sl) : nRangeBeg(sl.nRangeBeg), nRangeEnd(sl.nRangeEnd) {}	

	// Fusionne avec un autre scan si les intervalles se recoupent
	// Retourne true si la fusion a eu lieu
	inline bool MergeOnIntersection( const ScanLineRange& scan);

	// comparaison des dbuts d'intervalle pour savoir o insrer le scanline dans un tableau
	inline bool IsBefore( const ScanLineRange& scan) const;
	inline bool IsAfter( const ScanLineRange& scan) const  { return scan.IsBefore(*this); };
	inline bool DoesOverlap( const ScanLineRange& scan ) const { return !(IsBefore(scan) || IsAfter(scan));}

	inline bool operator < (const ScanLineRange& scan) const { return IsBefore(scan);}


};

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


// Classe definissant une ligne a partir d'un ensemble d'intervalles
class OneLineScansArray
{
	typedef std::vector<ScanLineRange> ScanLineArray;

public:
	inline OneLineScansArray(int32 nAlloc = 8) { m_aRanges.reserve(nAlloc); }

	// Accesseur lineaire d'intervalle en lecture seule
	inline const ScanLineRange& operator [] (int32 nIndex) const {return m_aRanges[nIndex];}

	inline int32 GetNbScans() const { return m_aRanges.size(); }

	// Reinitialisation de la ligne
	inline void ReInit() { m_aRanges.resize(0); }

	inline void Swap( OneLineScansArray& sa ) { m_aRanges.swap( sa.m_aRanges ); }

	// Ajout d'un intervalle en conservant la coherence de la ligne
	void AddScan( const ScanLineRange& range);

	void Affiche() const;

private:
ScanLineArray  m_aRanges;
};

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////

// Classe definissant une "surface" discrete en considerant un tableau de lignes

class AllScansArray
{

public:
	AllScansArray();
	int32 GetSize() const {return m_aScanLines.size(); }
	void  SetSize(int32 nSize)  { m_aScanLines.resize(nSize); ReInit(); }

	inline void Swap( AllScansArray& obj ) { m_slLines.Swap( obj.m_slLines); m_aScanLines.swap( obj.m_aScanLines);}


	const OneLineScansArray& operator[] (int32 nIdx) const { return m_aScanLines[nIdx];}
	const OneLineScansArray& GetLinesScans() const { return m_slLines;}


	void AddRectangle( const ScanLineRange& rangeX, const ScanLineRange& rangeY);
	void ReInit();
private:

	OneLineScansArray                 m_slLines;   // Scanline definissant les index de scanlines non vides
	std::vector<OneLineScansArray>    m_aScanLines;

};


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


inline bool ScanLineRange::IsBefore( const ScanLineRange& scan ) const
{
	return ( nRangeEnd+1 < scan.nRangeBeg );
}


inline bool ScanLineRange::MergeOnIntersection( const ScanLineRange& scan) 
{
	// Verification que le scan est coherent
	assert( scan.nRangeBeg <= scan.nRangeEnd );

	if( ! DoesOverlap(scan) )
		return false;
	if( nRangeBeg > scan.nRangeBeg)
		nRangeBeg = scan.nRangeBeg;

	if( nRangeEnd < scan.nRangeEnd)
		nRangeEnd = scan.nRangeEnd;
	return true;
}






} // namespace


#endif // #ifndef V3D_BLOBS_SCANLINES_H_INCLUDED
